/*
 * aes_vectors.h
 * AES test vectors
 *
 * Copyright 2003, Broadcom Corporation
 * All Rights Reserved.
 *
 * This is UNPUBLISHED PROPRIETARY SOURCE CODE of Broadcom Corporation; the
 * contents of this file may not be disclosed to third parties, copied or
 * duplicated in any form, in whole or in part, without the prior written
 * permission of Broadcom Corporation.
 *
 * $Id: aes_vectors.h 241182 2011-02-17 21:50:03Z $
 */

#include <typedefs.h>

/* Test vector data from NIST Special Publication 800-38A */

/* F.2.1 CBC-AES128.Encrypt and F.2.2 CBC-AES128.Decrypt */
uint8 key_00[] = {
	0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
	0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
	};

uint8 nonce_00[] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
	};

uint8 input_00[] = {
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
	0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
	0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
	0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
	0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
	};

uint8 ref_00[] = {
	0x76, 0x49, 0xab, 0xac, 0x81, 0x19, 0xb2, 0x46,
	0xce, 0xe9, 0x8e, 0x9b, 0x12, 0xe9, 0x19, 0x7d,
	0x50, 0x86, 0xcb, 0x9b, 0x50, 0x72, 0x19, 0xee,
	0x95, 0xdb, 0x11, 0x3a, 0x91, 0x76, 0x78, 0xb2,
	0x73, 0xbe, 0xd6, 0xb8, 0xe3, 0xc1, 0x74, 0x3b,
	0x71, 0x16, 0xe6, 0x9e, 0x22, 0x22, 0x95, 0x16,
	0x3f, 0xf1, 0xca, 0xa1, 0x68, 0x1f, 0xac, 0x09,
	0x12, 0x0e, 0xca, 0x30, 0x75, 0x86, 0xe1, 0xa7
	};


/* F.2.3 CBC-AES192.Encrypt and F.2.4 CBC-AES192.Decrypt */
uint8 key_01[] = {
	0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52,
	0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5,
	0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b
	};

uint8 nonce_01[] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
	};

uint8 input_01[] = {
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
	0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
	0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
	0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
	0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
	};

uint8 ref_01[] = {
	0x4f, 0x02, 0x1d, 0xb2, 0x43, 0xbc, 0x63, 0x3d,
	0x71, 0x78, 0x18, 0x3a, 0x9f, 0xa0, 0x71, 0xe8,
	0xb4, 0xd9, 0xad, 0xa9, 0xad, 0x7d, 0xed, 0xf4,
	0xe5, 0xe7, 0x38, 0x76, 0x3f, 0x69, 0x14, 0x5a,
	0x57, 0x1b, 0x24, 0x20, 0x12, 0xfb, 0x7a, 0xe0,
	0x7f, 0xa9, 0xba, 0xac, 0x3d, 0xf1, 0x02, 0xe0,
	0x08, 0xb0, 0xe2, 0x79, 0x88, 0x59, 0x88, 0x81,
	0xd9, 0x20, 0xa9, 0xe6, 0x4f, 0x56, 0x15, 0xcd
	};


/* F.2.5 CBC-AES256.Encrypt and F.2.6 CBC-AES256.Decrypt */
uint8 key_02[] = {
	0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe,
	0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7,
	0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4
	};

uint8 nonce_02[] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
	};

uint8 input_02[] = {
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
	0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
	0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
	0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
	0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
	};

uint8 ref_02[] = {
	0xf5, 0x8c, 0x4c, 0x04, 0xd6, 0xe5, 0xf1, 0xba,
	0x77, 0x9e, 0xab, 0xfb, 0x5f, 0x7b, 0xfb, 0xd6,
	0x9c, 0xfc, 0x4e, 0x96, 0x7e, 0xdb, 0x80, 0x8d,
	0x67, 0x9f, 0x77, 0x7b, 0xc6, 0x70, 0x2c, 0x7d,
	0x39, 0xf2, 0x33, 0x69, 0xa9, 0xd9, 0xba, 0xcf,
	0xa5, 0x30, 0xe2, 0x63, 0x04, 0x23, 0x14, 0x61,
	0xb2, 0xeb, 0x05, 0xe2, 0xc3, 0x9b, 0xe9, 0xfc,
	0xda, 0x6c, 0x19, 0x07, 0x8c, 0x6a, 0x9d, 0x1b
	};


/* F.5.1 CTR-AES128.Encrypt and F.5.2 CTR-AES128.Decrypt */
uint8 key_03[] = {
	0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
	0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
	};

uint8 nonce_03[] = {
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff
	};

uint8 input_03[] = {
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
	0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
	0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
	0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
	0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
	};

uint8 ref_03[] = {
	0x87, 0x4d, 0x61, 0x91, 0xb6, 0x20, 0xe3, 0x26,
	0x1b, 0xef, 0x68, 0x64, 0x99, 0x0d, 0xb6, 0xce,
	0x98, 0x06, 0xf6, 0x6b, 0x79, 0x70, 0xfd, 0xff,
	0x86, 0x17, 0x18, 0x7b, 0xb9, 0xff, 0xfd, 0xff,
	0x5a, 0xe4, 0xdf, 0x3e, 0xdb, 0xd5, 0xd3, 0x5e,
	0x5b, 0x4f, 0x09, 0x02, 0x0d, 0xb0, 0x3e, 0xab,
	0x1e, 0x03, 0x1d, 0xda, 0x2f, 0xbe, 0x03, 0xd1,
	0x79, 0x21, 0x70, 0xa0, 0xf3, 0x00, 0x9c, 0xee
	};


/* F.5.3 CTR-AES192.Encrypt and F.5.4 CTR-AES192.Decrypt */
uint8 key_04[] = {
	0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52,
	0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5,
	0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b
	};

uint8 nonce_04[] = {
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff
	};

uint8 input_04[] = {
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
	0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
	0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
	0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
	0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
	};

uint8 ref_04[] = {
	0x1a, 0xbc, 0x93, 0x24, 0x17, 0x52, 0x1c, 0xa2,
	0x4f, 0x2b, 0x04, 0x59, 0xfe, 0x7e, 0x6e, 0x0b,
	0x09, 0x03, 0x39, 0xec, 0x0a, 0xa6, 0xfa, 0xef,
	0xd5, 0xcc, 0xc2, 0xc6, 0xf4, 0xce, 0x8e, 0x94,
	0x1e, 0x36, 0xb2, 0x6b, 0xd1, 0xeb, 0xc6, 0x70,
	0xd1, 0xbd, 0x1d, 0x66, 0x56, 0x20, 0xab, 0xf7,
	0x4f, 0x78, 0xa7, 0xf6, 0xd2, 0x98, 0x09, 0x58,
	0x5a, 0x97, 0xda, 0xec, 0x58, 0xc6, 0xb0, 0x50
	};

/* F.5.5 CTR-AES256.Encrypt and F.5.6 CTR-AES256.Decrypt */
uint8 key_05[] = {
	0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe,
	0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7,
	0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4
	};

uint8 nonce_05[] = {
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff
	};

uint8 input_05[] = {
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
	0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
	0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
	0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
	0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
	};

uint8 ref_05[] = {
	0x60, 0x1e, 0xc3, 0x13, 0x77, 0x57, 0x89, 0xa5,
	0xb7, 0xa7, 0xf5, 0x04, 0xbb, 0xf3, 0xd2, 0x28,
	0xf4, 0x43, 0xe3, 0xca, 0x4d, 0x62, 0xb5, 0x9a,
	0xca, 0x84, 0xe9, 0x90, 0xca, 0xca, 0xf5, 0xc5,
	0x2b, 0x09, 0x30, 0xda, 0xa2, 0x3d, 0xe9, 0x4c,
	0xe8, 0x70, 0x17, 0xba, 0x2d, 0x84, 0x98, 0x8d,
	0xdf, 0xc9, 0xc5, 0x8d, 0xb6, 0x7a, 0xad, 0xa6,
	0x13, 0xc2, 0xdd, 0x08, 0x45, 0x79, 0x41, 0xa6
	};


typedef struct {
	int kl;
	uint8 *key;
	uint8 *nonce;
	int il;
	uint8 *input;
	uint8 *ref;
} aes_vector_t;

#define AES_VECTOR_ENTRY(x)	\
	{ sizeof(key_##x), key_##x, nonce_##x, sizeof(input_##x), input_##x, ref_##x }

aes_vector_t aes_cbc_vec[] = {
	AES_VECTOR_ENTRY(00),
	AES_VECTOR_ENTRY(01),
	AES_VECTOR_ENTRY(02),
	};
#define NUM_CBC_VECTORS	(sizeof(aes_cbc_vec)/sizeof(aes_cbc_vec[0]))

aes_vector_t aes_ctr_vec[] = {
	AES_VECTOR_ENTRY(03),
	AES_VECTOR_ENTRY(04),
	AES_VECTOR_ENTRY(05)
	};
#define NUM_CTR_VECTORS	(sizeof(aes_ctr_vec)/sizeof(aes_ctr_vec[0]))


/* Test vector data from: RFC3610 Counter with CBC-MAC (CCM) */

/* Packet Vector #1 */
uint8 key_06[] = {
	0xC0, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7,
	0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF
	};

uint8 nonce_06[] = {
	0x00, 0x00, 0x00, 0x03, 0x02, 0x01, 0x00, 0xA0,
	0xA1, 0xA2, 0xA3, 0xA4, 0xA5
	};

uint8 aad_06[] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07
	};

uint8 input_06[] = {
	0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
	0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E
	};

uint8 mac_06[] = {
	0x2D, 0xC6, 0x97, 0xE4, 0x11, 0xCA, 0x83, 0xA8
	};

uint8 ref_06[] = {
	0x58, 0x8C, 0x97, 0x9A, 0x61, 0xC6, 0x63, 0xD2,
	0xF0, 0x66, 0xD0, 0xC2, 0xC0, 0xF9, 0x89, 0x80,
	0x6D, 0x5F, 0x6B, 0x61, 0xDA, 0xC3, 0x84, 0x17,
	0xE8, 0xD1, 0x2C, 0xFD, 0xF9, 0x26, 0xE0
	};

/* Packet Vector #2 */
uint8 key_07[] = {
	0xC0, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7,
	0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF
	};

uint8 nonce_07[] = {
	0x00, 0x00, 0x00, 0x04, 0x03, 0x02, 0x01, 0xA0,
	0xA1, 0xA2, 0xA3, 0xA4, 0xA5
	};

uint8 aad_07[] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07
	};

uint8 input_07[] = {
	0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
	0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F
	};

uint8 mac_07[] = {
	0xF7, 0xB9, 0x05, 0x6A, 0x86, 0x92, 0x6C, 0xF3
	};

uint8 ref_07[] = {
	0x72, 0xC9, 0x1A, 0x36, 0xE1, 0x35, 0xF8, 0xCF,
	0x29, 0x1C, 0xA8, 0x94, 0x08, 0x5C, 0x87, 0xE3,
	0xCC, 0x15, 0xC4, 0x39, 0xC9, 0xE4, 0x3A, 0x3B,
	0xA0, 0x91, 0xD5, 0x6E, 0x10, 0x40, 0x09, 0x16
	};

/* Packet Vector #3 */
uint8 key_08[] = {
	0xC0, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7,
	0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF
	};

uint8 nonce_08[] = {
	0x00, 0x00, 0x00, 0x05, 0x04, 0x03, 0x02, 0xA0,
	0xA1, 0xA2, 0xA3, 0xA4, 0xA5
	};

uint8 aad_08[] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07
	};

uint8 input_08[] = {
	0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
	0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
	0x20
	};

uint8 mac_08[] = {
	0x73, 0xB4, 0x67, 0x75, 0xC0, 0x26, 0xDE, 0xAA
	};

uint8 ref_08[] = {
	0x51, 0xB1, 0xE5, 0xF4, 0x4A, 0x19, 0x7D, 0x1D,
	0xA4, 0x6B, 0x0F, 0x8E, 0x2D, 0x28, 0x2A, 0xE8,
	0x71, 0xE8, 0x38, 0xBB, 0x64, 0xDA, 0x85, 0x96,
	0x57, 0x4A, 0xDA, 0xA7, 0x6F, 0xBD, 0x9F, 0xB0,
	0xC5
	};

/* Packet Vector #4 */
uint8 key_09[] = {
	0xC0, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7,
	0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF
	};

uint8 nonce_09[] = {
	0x00, 0x00, 0x00, 0x06, 0x05, 0x04, 0x03, 0xA0,
	0xA1, 0xA2, 0xA3, 0xA4, 0xA5
	};

uint8 aad_09[] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0A, 0x0B
	};

uint8 input_09[] = {
	0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13,
	0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B,
	0x1C, 0x1D, 0x1E
	};

uint8 mac_09[] = {
	0x4B, 0x4F, 0x4B, 0x39, 0xB5, 0x93, 0xE6, 0xBF
	};

uint8 ref_09[] = {
	0xA2, 0x8C, 0x68, 0x65, 0x93, 0x9A, 0x9A, 0x79,
	0xFA, 0xAA, 0x5C, 0x4C, 0x2A, 0x9D, 0x4A, 0x91,
	0xCD, 0xAC, 0x8C, 0x96, 0xC8, 0x61, 0xB9, 0xC9,
	0xE6, 0x1E, 0xF1
	};

/* Packet Vector #5 */
uint8 key_10[] = {
	0xC0, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7,
	0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF
	};

uint8 nonce_10[] = {
	0x00, 0x00, 0x00, 0x07, 0x06, 0x05, 0x04, 0xA0,
	0xA1, 0xA2, 0xA3, 0xA4, 0xA5
	};

uint8 aad_10[] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0A, 0x0B
	};

uint8 input_10[] = {
	0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13,
	0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B,
	0x1C, 0x1D, 0x1E, 0x1F
	};

uint8 mac_10[] = {
	0x48, 0xB9, 0xE8, 0x82, 0x55, 0x05, 0x4A, 0xB5
	};

uint8 ref_10[] = {
	0xDC, 0xF1, 0xFB, 0x7B, 0x5D, 0x9E, 0x23, 0xFB,
	0x9D, 0x4E, 0x13, 0x12, 0x53, 0x65, 0x8A, 0xD8,
	0x6E, 0xBD, 0xCA, 0x3E, 0x51, 0xE8, 0x3F, 0x07,
	0x7D, 0x9C, 0x2D, 0x93
	};

/* Packet Vector #6 */
uint8 key_11[] = {
	0xC0, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7,
	0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF
	};

uint8 nonce_11[] = {
	0x00, 0x00, 0x00, 0x08, 0x07, 0x06, 0x05, 0xA0,
	0xA1, 0xA2, 0xA3, 0xA4, 0xA5
	};

uint8 aad_11[] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0A, 0x0B
	};

uint8 input_11[] = {
	0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13,
	0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B,
	0x1C, 0x1D, 0x1E, 0x1F, 0x20
	};

uint8 mac_11[] = {
	0xA5, 0x27, 0xD8, 0x15, 0x6A, 0xC3, 0x59, 0xBF
	};

uint8 ref_11[] = {
	0x6F, 0xC1, 0xB0, 0x11, 0xF0, 0x06, 0x56, 0x8B,
	0x51, 0x71, 0xA4, 0x2D, 0x95, 0x3D, 0x46, 0x9B,
	0x25, 0x70, 0xA4, 0xBD, 0x87, 0x40, 0x5A, 0x04,
	0x43, 0xAC, 0x91, 0xCB, 0x94
	};


typedef struct {
	int kl;
	uint8 *key;
	uint8 *nonce;
	int al;
	uint8 *aad;
	int il;
	uint8 *input;
	uint8 *mac;
	uint8 *ref;
} aes_ccm_vector_t;

#define AES_CCM_VECTOR_ENTRY(x)	\
{ \
	sizeof(key_##x), key_##x, nonce_##x, sizeof(aad_##x), aad_##x, \
	sizeof(input_##x), input_##x, mac_##x, ref_##x \
}

aes_ccm_vector_t aes_ccm_vec[] = {
	AES_CCM_VECTOR_ENTRY(06),
	AES_CCM_VECTOR_ENTRY(07),
	AES_CCM_VECTOR_ENTRY(08),
	AES_CCM_VECTOR_ENTRY(09),
	AES_CCM_VECTOR_ENTRY(10),
	AES_CCM_VECTOR_ENTRY(11),
	AES_CCM_VECTOR_ENTRY(99)
	};
#define NUM_CCM_VECTORS	(sizeof(aes_ccm_vec)/sizeof(aes_ccm_vec[0]))


/* Test vectors from 802.11i Draft 7.0 */
/* CCMP test mpdu 1 - works for both legacy and new ucode, so try it both ways */
uint8 key_20[] = {
	0xc9, 0x7c, 0x1f, 0x67, 0xce, 0x37, 0x11, 0x85,
	0x51, 0x4a, 0x8a, 0x19, 0xf2, 0xbd, 0xd5, 0x2f
	};

uint8 input_20[] = {
	0x08, 0x48, 0xc3, 0x2c, 0x0f, 0xd2, 0xe1, 0x28,
	0xa5, 0x7c, 0x50, 0x30, 0xf1, 0x84, 0x44, 0x08,
	0xab, 0xae, 0xa5, 0xb8, 0xfc, 0xba, 0x80, 0x33,
	0x0c, 0xe7, 0x00, 0x20, 0x76, 0x97, 0x03, 0xb5,
	0xf8, 0xba, 0x1a, 0x55, 0xd0, 0x2f, 0x85, 0xae,
	0x96, 0x7b, 0xb6, 0x2f, 0xb6, 0xcd, 0xa8, 0xeb,
	0x7e, 0x78, 0xa0, 0x50
	};

uint8 ref_20[] = {
	0x08, 0x48, 0xc3, 0x2c, 0x0f, 0xd2, 0xe1, 0x28,
	0xa5, 0x7c, 0x50, 0x30, 0xf1, 0x84, 0x44, 0x08,
	0xab, 0xae, 0xa5, 0xb8, 0xfc, 0xba, 0x80, 0x33,
	0x0c, 0xe7, 0x00, 0x20, 0x76, 0x97, 0x03, 0xb5,
	0xf3, 0xd0, 0xa2, 0xfe, 0x9a, 0x3d, 0xbf, 0x23,
	0x42, 0xa6, 0x43, 0xe4, 0x32, 0x46, 0xe8, 0x0c,
	0x3c, 0x04, 0xd0, 0x19, 0x78, 0x45, 0xce, 0x0b,
	0x16, 0xf9, 0x76, 0x23
	};

/* WDS, WME, legacy */
bool flags_20[] = { FALSE, FALSE, TRUE };

uint8 key_21[] = {
	0xc9, 0x7c, 0x1f, 0x67, 0xce, 0x37, 0x11, 0x85,
	0x51, 0x4a, 0x8a, 0x19, 0xf2, 0xbd, 0xd5, 0x2f
	};

uint8 input_21[] = {
	0x08, 0x48, 0xc3, 0x2c, 0x0f, 0xd2, 0xe1, 0x28,
	0xa5, 0x7c, 0x50, 0x30, 0xf1, 0x84, 0x44, 0x08,
	0xab, 0xae, 0xa5, 0xb8, 0xfc, 0xba, 0x80, 0x33,
	0x0c, 0xe7, 0x00, 0x20, 0x76, 0x97, 0x03, 0xb5,
	0xf8, 0xba, 0x1a, 0x55, 0xd0, 0x2f, 0x85, 0xae,
	0x96, 0x7b, 0xb6, 0x2f, 0xb6, 0xcd, 0xa8, 0xeb,
	0x7e, 0x78, 0xa0, 0x50
	};

uint8 ref_21[] = {
	0x08, 0x48, 0xc3, 0x2c, 0x0f, 0xd2, 0xe1, 0x28,
	0xa5, 0x7c, 0x50, 0x30, 0xf1, 0x84, 0x44, 0x08,
	0xab, 0xae, 0xa5, 0xb8, 0xfc, 0xba, 0x80, 0x33,
	0x0c, 0xe7, 0x00, 0x20, 0x76, 0x97, 0x03, 0xb5,
	0xf3, 0xd0, 0xa2, 0xfe, 0x9a, 0x3d, 0xbf, 0x23,
	0x42, 0xa6, 0x43, 0xe4, 0x32, 0x46, 0xe8, 0x0c,
	0x3c, 0x04, 0xd0, 0x19, 0x78, 0x45, 0xce, 0x0b,
	0x16, 0xf9, 0x76, 0x23
	};

/* WDS, WME, legacy */
bool flags_21[] = { FALSE, FALSE, FALSE };

/* CCMP test mpdu 2 */
uint8 key_22[] = {
	0x8f, 0x7a, 0x05, 0x3f, 0xa5, 0x77, 0xa5, 0x59,
	0x75, 0x29, 0x27, 0x20, 0x97, 0xa6, 0x03, 0xd5
	};

uint8 input_22[] = {
	0x38, 0xc0, 0x6a, 0x51, 0xea, 0x10, 0x0c, 0x84,
	0x68, 0x50, 0xee, 0xc1, 0x76, 0x2c, 0x88, 0xde,
	0xaf, 0x2e, 0xe9, 0xf4, 0x6a, 0x07, 0xe0, 0xcc,
	0xea, 0x97, 0x00, 0xa0, 0xba, 0xcb, 0xf3, 0x31,
	0x83, 0xa0, 0x63, 0x4b, 0x5e, 0xd7, 0x62, 0x7e,
	0xb9, 0xdf, 0x22, 0x5e, 0x05, 0x74, 0x03, 0x42,
	0xde, 0x19, 0x41, 0x17
	};

uint8 ref_22[] = {
	0x38, 0xc0, 0x6a, 0x51, 0xea, 0x10, 0x0c, 0x84,
	0x68, 0x50, 0xee, 0xc1, 0x76, 0x2c, 0x88, 0xde,
	0xaf, 0x2e, 0xe9, 0xf4, 0x6a, 0x07, 0xe0, 0xcc,
	0xea, 0x97, 0x00, 0xa0, 0xba, 0xcb, 0xf3, 0x31,
	0x81, 0x4b, 0x69, 0x65, 0xd0, 0x5b, 0xf2, 0xb2,
	0xed, 0x38, 0xd4, 0xbe, 0xb0, 0x69, 0xfe, 0x82,
	0x71, 0x4a, 0x61, 0x0b, 0x54, 0x2f, 0xbf, 0x8d,
	0xa0, 0x6a, 0xa4, 0xae
	};

bool flags_22[] = { FALSE, FALSE, FALSE };

/* CCMP test mpdu 3 */
uint8 key_23[] = {
	0x40, 0xcf, 0xb7, 0xa6, 0x2e, 0x88, 0x01, 0x3b,
	0xd6, 0xd3, 0xaf, 0xfc, 0xc1, 0x91, 0x04, 0x1e
	};

uint8 input_23[] = {
	0xb8, 0xc8, 0xdc, 0x61, 0xd9, 0x57, 0x7d, 0xf7,
	0x63, 0xc8, 0xb6, 0xa8, 0x8a, 0xdf, 0x36, 0x91,
	0xdc, 0x4a, 0x8b, 0xca, 0x94, 0xdd, 0x60, 0x82,
	0x20, 0x85, 0xae, 0xa5, 0x00, 0xa0, 0xf3, 0xa0,
	0xdc, 0x2f, 0x2c, 0x1b, 0xd0, 0x36, 0x83, 0x1c,
	0x95, 0x49, 0x6c, 0x5f, 0x4d, 0xbf, 0x3d, 0x55,
	0x9e, 0x72, 0xde, 0x80, 0x2a, 0x18
	};

uint8 ref_23[] = {
	0xb8, 0xc8, 0xdc, 0x61, 0xd9, 0x57, 0x7d, 0xf7,
	0x63, 0xc8, 0xb6, 0xa8, 0x8a, 0xdf, 0x36, 0x91,
	0xdc, 0x4a, 0x8b, 0xca, 0x94, 0xdd, 0x60, 0x82,
	0x20, 0x85, 0xae, 0xa5, 0x00, 0xa0, 0xf3, 0xa0,
	0xdc, 0x2f, 0x89, 0xd8, 0x58, 0x03, 0x40, 0xb6,
	0x26, 0xa0, 0xb6, 0xd4, 0xd0, 0x13, 0xbf, 0x18,
	0xf2, 0x91, 0xb8, 0x96, 0x46, 0xc8, 0xfd, 0x1f,
	0x1f, 0x61, 0xa9, 0xfb, 0x4b, 0xb3
	};

bool flags_23[] = { FALSE, TRUE, FALSE };

/* CCMP test mpdu 4 */
uint8 key_24[] = {
	0x8c, 0x89, 0xa2, 0xeb, 0xc9, 0x6c, 0x76, 0x02,
	0x70, 0x7f, 0xcf, 0x24, 0xb3, 0x2d, 0x38, 0x33
	};

uint8 input_24[] = {
	0xa8, 0xca, 0x3a, 0x11, 0x71, 0x2a, 0x9d, 0xdf,
	0x11, 0xdb, 0x8e, 0xf8, 0x22, 0x73, 0x47, 0x01,
	0x59, 0x14, 0x0d, 0xd6, 0x46, 0xa2, 0xc0, 0x2f,
	0x67, 0xa5, 0xe3, 0x0f, 0x00, 0x20, 0x5a, 0xa5,
	0x70, 0xf6, 0x4f, 0xad, 0x2b, 0x1c, 0x29, 0x0f,
	0xa5, 0xeb, 0xd8, 0x72, 0xfb, 0xc3, 0xf3, 0xa0,
	0x74, 0x89, 0x8f, 0x8b, 0x2f, 0xbb
	};

uint8 ref_24[] = {
	0xa8, 0xca, 0x3a, 0x11, 0x71, 0x2a, 0x9d, 0xdf,
	0x11, 0xdb, 0x8e, 0xf8, 0x22, 0x73, 0x47, 0x01,
	0x59, 0x14, 0x0d, 0xd6, 0x46, 0xa2, 0xc0, 0x2f,
	0x67, 0xa5, 0xe3, 0x0f, 0x00, 0x20, 0x5a, 0xa5,
	0x70, 0xf6, 0x9d, 0x59, 0xb1, 0x5f, 0x37, 0x14,
	0x48, 0xc2, 0x30, 0xf4, 0xd7, 0x39, 0x05, 0x2e,
	0x13, 0xab, 0x3b, 0x1a, 0x7b, 0x10, 0x31, 0xfc,
	0x88, 0x00, 0x4f, 0x35, 0xee, 0x3d
	};

bool flags_24[] = { FALSE, TRUE, FALSE };

/* CCMP test mpdu 5 */
uint8 key_25[] = {
	0xa5, 0x74, 0xd5, 0x14, 0x3b, 0xb2, 0x5e, 0xfd,
	0xde, 0xff, 0x30, 0x12, 0x2f, 0xdf, 0xd0, 0x66
	};

uint8 input_25[] = {
	0x88, 0xda, 0x18, 0x41, 0x45, 0xde, 0xc6, 0x9a,
	0x74, 0x80, 0xf3, 0x51, 0x94, 0x6b, 0xc9, 0x6b,
	0xe2, 0x76, 0xfb, 0xe6, 0xc1, 0x27, 0x80, 0xf2,
	0x4b, 0x19, 0x37, 0x0e, 0x00, 0xa0, 0x3c, 0xe0,
	0xff, 0xa7, 0x28, 0x96, 0x9b, 0x95, 0x4f, 0x26,
	0x3a, 0x80, 0x18, 0xa9, 0xef, 0x70, 0xa8, 0xb0,
	0x51, 0x46, 0x24, 0x81, 0x92, 0x2e
	};

uint8 ref_25[] = {
	0x88, 0xda, 0x18, 0x41, 0x45, 0xde, 0xc6, 0x9a,
	0x74, 0x80, 0xf3, 0x51, 0x94, 0x6b, 0xc9, 0x6b,
	0xe2, 0x76, 0xfb, 0xe6, 0xc1, 0x27, 0x80, 0xf2,
	0x4b, 0x19, 0x37, 0x0e, 0x00, 0xa0, 0x3c, 0xe0,
	0xff, 0xa7, 0xeb, 0x4a, 0xe4, 0x95, 0x6a, 0x80,
	0x1d, 0xa9, 0x62, 0x4b, 0x7e, 0x0c, 0x18, 0xb2,
	0x3e, 0x61, 0x5e, 0xc0, 0x3a, 0xf6, 0xce, 0x0c,
	0x3b, 0xe1, 0x97, 0xd3, 0x05, 0xeb
	};

bool flags_25[] = { FALSE, TRUE, FALSE };

/* CCMP test mpdu 6 */
uint8 key_26[] = {
	0xf7, 0x1e, 0xea, 0x4e, 0x1f, 0x58, 0x80, 0x4b,
	0x97, 0x17, 0x23, 0x0a, 0xd0, 0x61, 0x46, 0x41
	};

uint8 input_26[] = {
	0x88, 0x52, 0xe1, 0x1f, 0x5a, 0xf2, 0x84, 0x30,
	0xfd, 0xab, 0xbf, 0xf9, 0x43, 0xb9, 0xf9, 0xa6,
	0xab, 0x1d, 0x98, 0xc7, 0xfe, 0x73, 0x50, 0x71,
	0x3d, 0x6a, 0x89, 0x89, 0x00, 0x60, 0xa4, 0xec,
	0x81, 0x6b, 0xab, 0xfd, 0xa2, 0x2d, 0x3a, 0x0b,
	0xfc, 0x9c, 0xc1, 0xfc, 0x07, 0x93, 0x63, 0xc2,
	0xfc, 0xa1, 0x43, 0xe6, 0xeb, 0x1d
	};

uint8 ref_26[] = {
	0x88, 0x52, 0xe1, 0x1f, 0x5a, 0xf2, 0x84, 0x30,
	0xfd, 0xab, 0xbf, 0xf9, 0x43, 0xb9, 0xf9, 0xa6,
	0xab, 0x1d, 0x98, 0xc7, 0xfe, 0x73, 0x50, 0x71,
	0x3d, 0x6a, 0x89, 0x89, 0x00, 0x60, 0xa4, 0xec,
	0x81, 0x6b, 0x9a, 0x70, 0x9b, 0x60, 0xa3, 0x9d,
	0x40, 0xb1, 0xdf, 0xb6, 0x12, 0xe1, 0x8b, 0x5f,
	0x11, 0x4b, 0xad, 0xb6, 0xcc, 0x86, 0x30, 0x9a,
	0x8d, 0x5c, 0x46, 0x6b, 0xbb, 0x71
	};

bool flags_26[] = { FALSE, TRUE, FALSE };

/* CCMP test mpdu 7 */
uint8 key_27[] = {
	0x1b, 0xdb, 0x34, 0x98, 0x0e, 0x03, 0x81, 0x24,
	0xa1, 0xdb, 0x1a, 0x89, 0x2b, 0xec, 0x36, 0x6a
	};

uint8 input_27[] = {
	0x18, 0x79, 0x81, 0x46, 0x9b, 0x50, 0xf4, 0xfd,
	0x56, 0xf6, 0xef, 0xec, 0x95, 0x20, 0x16, 0x91,
	0x83, 0x57, 0x0c, 0x4c, 0xcd, 0xee, 0x20, 0xa0,
	0x23, 0xe7, 0x00, 0xe0, 0x73, 0x40, 0xec, 0x5e,
	0x98, 0xbe, 0xca, 0x86, 0xf4, 0xb3, 0x8d, 0xa2,
	0x0c, 0xfd, 0xf2, 0x47, 0x24, 0xc5, 0x8e, 0xb8,
	0x35, 0x66, 0x53, 0x39
	};

uint8 ref_27[] = {
	0x18, 0x79, 0x81, 0x46, 0x9b, 0x50, 0xf4, 0xfd,
	0x56, 0xf6, 0xef, 0xec, 0x95, 0x20, 0x16, 0x91,
	0x83, 0x57, 0x0c, 0x4c, 0xcd, 0xee, 0x20, 0xa0,
	0x23, 0xe7, 0x00, 0xe0, 0x73, 0x40, 0xec, 0x5e,
	0x12, 0xc5, 0x37, 0xeb, 0xf3, 0xab, 0x58, 0x4e,
	0xf1, 0xfe, 0xf9, 0xa1, 0xf3, 0x54, 0x7a, 0x8c,
	0x13, 0xb3, 0x22, 0x5a, 0x2d, 0x09, 0x57, 0xec,
	0xfa, 0xbe, 0x95, 0xb9
	};

bool flags_27[] = { FALSE, FALSE, FALSE };

/* CCMP test mpdu 8 */
uint8 key_28[] = {
	0x6e, 0xac, 0x1b, 0xf5, 0x4b, 0xd5, 0x4e, 0xdb,
	0x23, 0x21, 0x75, 0x43, 0x03, 0x02, 0x4c, 0x71
	};

uint8 input_28[] = {
	0xb8, 0xd9, 0x4c, 0x72, 0x55, 0x2d, 0x5f, 0x72,
	0xbb, 0x70, 0xca, 0x3f, 0x3a, 0xae, 0x60, 0xc4,
	0x8b, 0xa9, 0xb5, 0xf8, 0x2c, 0x2f, 0x50, 0xeb,
	0x2a, 0x55, 0xdd, 0xcc, 0x00, 0xa0, 0x6e, 0x99,
	0xfd, 0xce, 0x57, 0xcb, 0x5c, 0x0e, 0x5f, 0xcd,
	0x88, 0x5e, 0x9a, 0x42, 0x39, 0xe9, 0xb9, 0xca,
	0xd6, 0x0d, 0x64, 0x37, 0x59, 0x79
	};

uint8 ref_28[] = {
	0xb8, 0xd9, 0x4c, 0x72, 0x55, 0x2d, 0x5f, 0x72,
	0xbb, 0x70, 0xca, 0x3f, 0x3a, 0xae, 0x60, 0xc4,
	0x8b, 0xa9, 0xb5, 0xf8, 0x2c, 0x2f, 0x50, 0xeb,
	0x2a, 0x55, 0xdd, 0xcc, 0x00, 0xa0, 0x6e, 0x99,
	0xfd, 0xce, 0x4b, 0xf2, 0x81, 0xef, 0x8e, 0xc7,
	0x73, 0x9f, 0x91, 0x59, 0x1b, 0x97, 0xa8, 0x7d,
	0xc1, 0x4b, 0x3f, 0xa1, 0x74, 0x62, 0x6d, 0xba,
	0x8e, 0xf7, 0xf0, 0x80, 0x87, 0xdd
	};

bool flags_28[] = { FALSE, TRUE, FALSE };

/* CCMP test mpdu 9 */
uint8 key_29[] = {
	0x49, 0x4b, 0x50, 0x1e, 0x19, 0x46, 0x75, 0x97,
	0x1a, 0x48, 0xd0, 0x8c, 0x5b, 0xc3, 0x53, 0xcb
	};

uint8 input_29[] = {
	0xa8, 0xf9, 0xc8, 0x42, 0x9a, 0xfb, 0x79, 0x8b,
	0x8a, 0x4b, 0xa4, 0xad, 0x6d, 0x31, 0x99, 0x85,
	0xbc, 0x42, 0x9e, 0x8f, 0x0a, 0xfa, 0xf0, 0x7d,
	0x4a, 0x80, 0xb3, 0x37, 0x00, 0xe0, 0x34, 0xe9,
	0x82, 0xba, 0x25, 0xa9, 0x8f, 0x9c, 0x1b, 0xd9,
	0xc9, 0x3c, 0xf3, 0x83, 0xab, 0x9d, 0x98, 0x15,
	0x2d, 0x76, 0xcb, 0x4a, 0x32, 0xc6
	};

uint8 ref_29[] = {
	0xa8, 0xf9, 0xc8, 0x42, 0x9a, 0xfb, 0x79, 0x8b,
	0x8a, 0x4b, 0xa4, 0xad, 0x6d, 0x31, 0x99, 0x85,
	0xbc, 0x42, 0x9e, 0x8f, 0x0a, 0xfa, 0xf0, 0x7d,
	0x4a, 0x80, 0xb3, 0x37, 0x00, 0xe0, 0x34, 0xe9,
	0x82, 0xba, 0x56, 0x1a, 0x0d, 0x06, 0x8e, 0xac,
	0x2e, 0xad, 0xb0, 0xc5, 0x7f, 0xe2, 0xd0, 0xa6,
	0xcc, 0x73, 0x98, 0xb6, 0xdd, 0xbf, 0xcf, 0xe4,
	0x38, 0xcb, 0xea, 0x61, 0xfa, 0x9a
	};

bool flags_29[] = { FALSE, TRUE, FALSE };

/* CCMP test mpdu 10 */
uint8 key_30[] = {
	0x48, 0x9e, 0x49, 0xbc, 0x3c, 0xfe, 0x3f, 0xce,
	0x38, 0x95, 0x82, 0x0e, 0x87, 0x2e, 0xe1, 0xa3
	};

uint8 input_30[] = {
	0x18, 0x4b, 0xe0, 0x63, 0x40, 0xec, 0x29, 0xfa,
	0x75, 0x9b, 0x53, 0xf8, 0x69, 0xfe, 0x27, 0x9a,
	0xf0, 0xf9, 0xf8, 0xa6, 0x54, 0x16, 0x80, 0xe3,
	0x52, 0xbf, 0xd2, 0x70, 0x3d, 0x24, 0xa9, 0xfe,
	0x00, 0x20, 0x75, 0x3e, 0x1d, 0xcf, 0x7f, 0x91,
	0xf2, 0x47, 0x2d, 0x7a, 0x12, 0x1c, 0x9c, 0xdd,
	0x4b, 0x6c, 0x90, 0x80, 0x67, 0x5a, 0x10, 0x20,
	0xaa, 0x00
	};

uint8 ref_30[] = {
	0x18, 0x4b, 0xe0, 0x63, 0x40, 0xec, 0x29, 0xfa,
	0x75, 0x9b, 0x53, 0xf8, 0x69, 0xfe, 0x27, 0x9a,
	0xf0, 0xf9, 0xf8, 0xa6, 0x54, 0x16, 0x80, 0xe3,
	0x52, 0xbf, 0xd2, 0x70, 0x3d, 0x24, 0xa9, 0xfe,
	0x00, 0x20, 0x75, 0x3e, 0x1d, 0xcf, 0x25, 0xdf,
	0x51, 0x73, 0x83, 0x5e, 0x4f, 0xba, 0x23, 0xbc,
	0x05, 0xa2, 0x53, 0x88, 0x5e, 0xbe, 0xd3, 0xac,
	0x48, 0x71, 0xc8, 0x68, 0xa7, 0x25, 0x55, 0x2c,
	0x55, 0x65
	};

bool flags_30[] = { TRUE, FALSE, FALSE };

/* CCMP test mpdu 11 */
uint8 key_31[] = {
	0x02, 0xbe, 0x5c, 0x45, 0x45, 0x67, 0x2a, 0x07,
	0xe4, 0xe3, 0x14, 0xd7, 0x0f, 0x1f, 0x9e, 0x85
	};

uint8 input_31[] = {
	0xb8, 0xdb, 0x6a, 0x54, 0x29, 0x8c, 0x0b, 0xaa,
	0x91, 0x90, 0x34, 0x7c, 0xeb, 0x9a, 0xab, 0xff,
	0xd8, 0x3d, 0x48, 0x86, 0xe5, 0xc2, 0xe0, 0x53,
	0xe2, 0x9d, 0x52, 0x4a, 0xe1, 0x96, 0x3d, 0x62,
	0xd4, 0x55, 0x00, 0xe0, 0x6e, 0x59, 0xd6, 0xd2,
	0xf9, 0xa8, 0x12, 0xe4, 0xa2, 0x8a, 0xf7, 0xf3,
	0x71, 0x4d, 0x4b, 0xf6, 0x62, 0x2e, 0x59, 0x32,
	0xf2, 0x18, 0x45, 0x09
	};

uint8 ref_31[] = {
	0xb8, 0xdb, 0x6a, 0x54, 0x29, 0x8c, 0x0b, 0xaa,
	0x91, 0x90, 0x34, 0x7c, 0xeb, 0x9a, 0xab, 0xff,
	0xd8, 0x3d, 0x48, 0x86, 0xe5, 0xc2, 0xe0, 0x53,
	0xe2, 0x9d, 0x52, 0x4a, 0xe1, 0x96, 0x3d, 0x62,
	0xd4, 0x55, 0x00, 0xe0, 0x6e, 0x59, 0xd6, 0xd2,
	0x63, 0x15, 0x50, 0x0f, 0x92, 0x42, 0x95, 0xcd,
	0x3e, 0xaf, 0xbd, 0xc3, 0xe1, 0x51, 0xb1, 0xdf,
	0x46, 0x46, 0x5b, 0x71, 0x68, 0x1f, 0xde, 0xe8,
	0x51, 0x3c, 0x62, 0xdc
	};

bool flags_31[] = { TRUE, TRUE, FALSE };

/* CCMP test mpdu 12 */
uint8 key_32[] = {
	0x77, 0x07, 0x7e, 0xd7, 0x94, 0x53, 0xe4, 0xa1,
	0x8d, 0x60, 0x43, 0x8c, 0xc6, 0x48, 0x4d, 0x6e
	};

uint8 input_32[] = {
	0x18, 0x6b, 0xd0, 0x27, 0xaa, 0x28, 0x8b, 0x84,
	0x35, 0xbc, 0xd8, 0xac, 0x5a, 0x7e, 0xc4, 0x44,
	0xe8, 0xb4, 0x62, 0x50, 0x53, 0x8b, 0x80, 0x62,
	0xe8, 0x14, 0x02, 0xc2, 0xee, 0x11, 0x8e, 0xfd,
	0x00, 0xe0, 0x77, 0x1e, 0xb0, 0x50, 0x43, 0x19,
	0x81, 0xa2, 0x33, 0x6d, 0x02, 0xf8, 0xcb, 0x84,
	0x48, 0xd5, 0x42, 0x89, 0x16, 0xbe, 0x95, 0x29,
	0x35, 0x37
	};

uint8 ref_32[] = {
	0x18, 0x6b, 0xd0, 0x27, 0xaa, 0x28, 0x8b, 0x84,
	0x35, 0xbc, 0xd8, 0xac, 0x5a, 0x7e, 0xc4, 0x44,
	0xe8, 0xb4, 0x62, 0x50, 0x53, 0x8b, 0x80, 0x62,
	0xe8, 0x14, 0x02, 0xc2, 0xee, 0x11, 0x8e, 0xfd,
	0x00, 0xe0, 0x77, 0x1e, 0xb0, 0x50, 0xcf, 0x71,
	0xb2, 0xcc, 0xbd, 0x59, 0x0b, 0x20, 0x80, 0x07,
	0x92, 0xf3, 0x59, 0xed, 0x1c, 0xfd, 0x74, 0xd8,
	0x00, 0xb4, 0xfd, 0x0f, 0x41, 0xf4, 0x26, 0xbb,
	0x8f, 0x30
	};

bool flags_32[] = { TRUE, FALSE, FALSE };


/* Test vectors generated by legacy ucode */
uint8 key_40[] = {
	0x12, 0x34, 0x56, 0x78, 0x90, 0x12, 0x34, 0x56,
	0x78, 0x90, 0x12, 0x34, 0x56, 0x78, 0x90, 0x12
	};

uint8 input_40[] = {
	0x08, 0x40, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0x00, 0x10, 0x18, 0x90, 0x01, 0xee,
	0x56, 0xb9, 0x99, 0xdd, 0x40, 0x04, 0x30, 0x9f,
	0x07, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00,
	0xaa, 0xaa, 0x03, 0x00, 0x00, 0x00, 0x08, 0x06,
	0x00, 0x01, 0x08, 0x00, 0x06, 0x04, 0x00, 0x01,
	0x00, 0x10, 0x18, 0x90, 0x01, 0xee, 0xc0, 0xa8,
	0x0a, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xc0, 0xa8, 0x0a, 0x01
	};

uint8 ref_40[] = {
	0x08, 0x40, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0x00, 0x10, 0x18, 0x90, 0x01, 0xee,
	0x56, 0xb9, 0x99, 0xdd, 0x40, 0x04, 0x30, 0x9f,
	0x07, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00,
	0x85, 0x7d, 0x0c, 0xf3, 0xe7, 0x0b, 0x05, 0xb2,
	0x66, 0x2f, 0xf4, 0xe1, 0xc4, 0x86, 0x1f, 0x94,
	0x02, 0x3f, 0x25, 0x41, 0xcf, 0x65, 0xec, 0xd1,
	0x33, 0x70, 0x8f, 0x62, 0x98, 0xa8, 0x72, 0x5f,
	0x67, 0xe3, 0xaf, 0x78, 0xb1, 0x0d, 0x04, 0x08,
	0x95, 0x2e, 0x11, 0x7a
	};

bool flags_40[] = { FALSE, FALSE, TRUE };

uint8 key_41[] = {
	0x12, 0x34, 0x56, 0x78, 0x90, 0x12, 0x34, 0x56,
	0x78, 0x90, 0x12, 0x34, 0x56, 0x78, 0x90, 0x12
	};

uint8 input_41[] = {
	0x08, 0x40, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0x00, 0x10, 0x18, 0x90, 0x01, 0xee,
	0x56, 0xb9, 0x99, 0xdd, 0x40, 0x04, 0xb0, 0xa0,
	0x09, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00,
	0xaa, 0xaa, 0x03, 0x00, 0x00, 0x00, 0x08, 0x06,
	0x00, 0x01, 0x08, 0x00, 0x06, 0x04, 0x00, 0x01,
	0x00, 0x10, 0x18, 0x90, 0x01, 0xee, 0xc0, 0xa8,
	0x0a, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xc0, 0xa8, 0x0a, 0x01
	};

uint8 ref_41[] = {
	0x08, 0x40, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0x00, 0x10, 0x18, 0x90, 0x01, 0xee,
	0x56, 0xb9, 0x99, 0xdd, 0x40, 0x04, 0xb0, 0xa0,
	0x09, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00,
	0x0a, 0xd4, 0xf2, 0x74, 0x0e, 0x45, 0x6f, 0x6a,
	0xe2, 0x7f, 0xcd, 0x98, 0xa6, 0x54, 0x51, 0xb3,
	0x08, 0x7f, 0xf5, 0x68, 0xbe, 0xcf, 0xd0, 0x98,
	0xbc, 0xbd, 0x70, 0x59, 0xa9, 0xc8, 0xc0, 0x09,
	0xcb, 0x5f, 0x1d, 0x99, 0x71, 0x74, 0x5d, 0x39,
	0xe5, 0x52, 0x92, 0xb9
	};

bool flags_41[] = { FALSE, FALSE, TRUE };

uint8 key_42[] = {
	0x12, 0x34, 0x56, 0x78, 0x90, 0x12, 0x34, 0x56,
	0x78, 0x90, 0x12, 0x34, 0x56, 0x78, 0x90, 0x12
	};

uint8 input_42[] = {
	0x08, 0x40, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0x00, 0x10, 0x18, 0x90, 0x01, 0xee,
	0x56, 0xb9, 0x99, 0xdd, 0x40, 0x04, 0x70, 0xa1,
	0x0a, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00,
	0xaa, 0xaa, 0x03, 0x00, 0x00, 0x00, 0x08, 0x06,
	0x00, 0x01, 0x08, 0x00, 0x06, 0x04, 0x00, 0x01,
	0x00, 0x10, 0x18, 0x90, 0x01, 0xee, 0xc0, 0xa8,
	0x0a, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xc0, 0xa8, 0x0a, 0x01
	};

uint8 ref_42[] = {
	0x08, 0x40, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0x00, 0x10, 0x18, 0x90, 0x01, 0xee,
	0x56, 0xb9, 0x99, 0xdd, 0x40, 0x04, 0x70, 0xa1,
	0x0a, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00,
	0x73, 0xb2, 0x72, 0x41, 0xb8, 0x18, 0x10, 0x27,
	0xbc, 0x2e, 0x94, 0x4f, 0x86, 0x14, 0xfb, 0x7d,
	0x54, 0x85, 0xa1, 0x39, 0x52, 0xf0, 0x73, 0x94,
	0x05, 0x8c, 0xe8, 0xf0, 0xf8, 0xc8, 0x09, 0xfb,
	0x3b, 0x99, 0x3a, 0x1d, 0x49, 0xb1, 0xd1, 0x44,
	0xe4, 0x4d, 0xab, 0xaa
	};

bool flags_42[] = { FALSE, FALSE, TRUE };

uint8 key_43[] = {
	0x12, 0x34, 0x56, 0x78, 0x90, 0x12, 0x34, 0x56,
	0x78, 0x90, 0x12, 0x34, 0x56, 0x78, 0x90, 0x12
	};

uint8 input_43[] = {
	0x08, 0x40, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0x00, 0x10, 0x18, 0x90, 0x01, 0xee,
	0x56, 0xb9, 0x99, 0xdd, 0x40, 0x04, 0x20, 0xa2,
	0x0b, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00,
	0xaa, 0xaa, 0x03, 0x00, 0x00, 0x00, 0x08, 0x06,
	0x00, 0x01, 0x08, 0x00, 0x06, 0x04, 0x00, 0x01,
	0x00, 0x10, 0x18, 0x90, 0x01, 0xee, 0xc0, 0xa8,
	0x0a, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xc0, 0xa8, 0x0a, 0x01
	};

uint8 ref_43[] = {
	0x08, 0x40, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0x00, 0x10, 0x18, 0x90, 0x01, 0xee,
	0x56, 0xb9, 0x99, 0xdd, 0x40, 0x04, 0x20, 0xa2,
	0x0b, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00,
	0xcf, 0x1c, 0x69, 0xfb, 0x5d, 0x67, 0xcd, 0xc2,
	0x9a, 0x93, 0x13, 0xec, 0x71, 0xc8, 0xb2, 0xbb,
	0x3e, 0x25, 0x0e, 0x4c, 0x1d, 0xcb, 0x44, 0x41,
	0x83, 0x6b, 0xc1, 0xb3, 0x25, 0x1d, 0xb0, 0xb2,
	0x58, 0x90, 0x5e, 0xa3, 0x5d, 0x3c, 0xf9, 0xcd,
	0x98, 0x4c, 0xad, 0xd9
	};

bool flags_43[] = { FALSE, FALSE, TRUE };

uint8 key_44[] = {
	0x12, 0x34, 0x56, 0x78, 0x90, 0x12, 0x34, 0x56,
	0x78, 0x90, 0x12, 0x34, 0x56, 0x78, 0x90, 0x12,
	};

uint8 input_44[] = {
	0x08, 0x40, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0x00, 0x10, 0x18, 0x90, 0x01, 0xee,
	0x56, 0xb9, 0x99, 0xdd, 0x40, 0x04, 0xd0, 0xa2,
	0x0c, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00,
	0xaa, 0xaa, 0x03, 0x00, 0x00, 0x00, 0x08, 0x06,
	0x00, 0x01, 0x08, 0x00, 0x06, 0x04, 0x00, 0x01,
	0x00, 0x10, 0x18, 0x90, 0x01, 0xee, 0xc0, 0xa8,
	0x0a, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xc0, 0xa8, 0x0a, 0x01,
	};

uint8 ref_44[] = {
	0x08, 0x40, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0x00, 0x10, 0x18, 0x90, 0x01, 0xee,
	0x56, 0xb9, 0x99, 0xdd, 0x40, 0x04, 0xd0, 0xa2,
	0x0c, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00,
	0xa5, 0x8a, 0xcf, 0x80, 0xe0, 0xe9, 0x96, 0x1f,
	0x95, 0x9e, 0x71, 0x57, 0x7b, 0x4a, 0x77, 0x1d,
	0x15, 0xae, 0x47, 0xa6, 0x2a, 0xfb, 0x9a, 0x65,
	0x3f, 0x3a, 0x19, 0x76, 0x9f, 0x73, 0xfa, 0xa6,
	0xa7, 0x85, 0x6f, 0xb8, 0x5b, 0x3e, 0x54, 0x25,
	0x34, 0x00, 0xa8, 0x44
	};

bool flags_44[] = { FALSE, FALSE, TRUE };


typedef struct {
	int kl;
	uint8 *key;
	int il;
	uint8 *input;
	uint8 *ref;
	bool *flags;
} aes_ccmp_vector_t;

#define AES_CCMP_VECTOR_ENTRY(x)	\
	{ sizeof(key_##x), key_##x, sizeof(input_##x), input_##x, ref_##x, flags_##x }

aes_ccmp_vector_t aes_ccmp_vec[] = {
	AES_CCMP_VECTOR_ENTRY(20),
	AES_CCMP_VECTOR_ENTRY(21),
	AES_CCMP_VECTOR_ENTRY(22),
	AES_CCMP_VECTOR_ENTRY(23),
	AES_CCMP_VECTOR_ENTRY(24),
	AES_CCMP_VECTOR_ENTRY(25),
	AES_CCMP_VECTOR_ENTRY(26),
	AES_CCMP_VECTOR_ENTRY(27),
	AES_CCMP_VECTOR_ENTRY(28),
	AES_CCMP_VECTOR_ENTRY(29),
	AES_CCMP_VECTOR_ENTRY(30),
	AES_CCMP_VECTOR_ENTRY(31),
	AES_CCMP_VECTOR_ENTRY(32),
	AES_CCMP_VECTOR_ENTRY(40),
	AES_CCMP_VECTOR_ENTRY(41),
	AES_CCMP_VECTOR_ENTRY(42),
	AES_CCMP_VECTOR_ENTRY(43),
	AES_CCMP_VECTOR_ENTRY(44)
	};
#define NUM_CCMP_VECTORS	(sizeof(aes_ccmp_vec)/sizeof(aes_ccmp_vec[0]))
